/*
*******************************************************************************
*
* File:         mailbox.c
* Description:  Mailboxing functions
* Author:       Henrik Liebau, BVS R&D
* Created:      Wed Sep 15 15:07:14 1999
* Modified:     Tue Dec 18 14:50:02 2001 by Henrik Liebau, BVS R&D
* Language:     C
* Package:      HP E2920 Series CAPI
* Status:       
*
* (C) Copyright 1999, Hewlett-Packard GmbH, All Rights Reserved.
*
*******************************************************************************
*/

#ifdef WINNT
# include "IdentOs.h"
# include "pci32reg.h"
#endif  /* WINNT */

#include <typedefs.h>
#include <mailbox.h>

#include "pci.h"




#ifndef BEST_FIRMWARE

/*---------------------------------------------------------------------------*
 * b_errtype EXPORT BestDevIdentifierGet(
 *
 * Purpose	: Find BEST device on PCI bus
 * Inputs	: vendorid, deviceid, function number
 * Outputs	: devid
 * Returns	: B_E_OK on success, B_E_<errcode> otherwise
 *---------------------------------------------------------------------------*/
b_errtype EXPORT BestDevIdentifierGet(
    b_int32 vendor_id,
    b_int32 device_id,
    b_int32 number,
    b_int32 * devid
)
{

/* 
   DeviceID consists of BusNo(8bit),Slotnumber(5 bits) and functionnumber (3 bit):
   devid=busnumber<<8 | slotnumber<<3 | function
*/

#ifdef WINNT
  /* we can do this the easy way */
  return BestNTRegDevIDGet(vendor_id, device_id, number, devid);

#else
  
  /* we're in DOS or other non-Win32 land */

  b_int32 cNumFound = 0;
  b_int32 slotnumber;
  b_int32 value;
  b_int32 LookForValue = (vendor_id | (device_id << 16));
  b_errtype err;

  /* Ther are 256 possible busses, each containing 32 slots */
  for (slotnumber = 0; slotnumber < 0x2000UL; slotnumber++)
  {
    /* check vendor & device; located at offset zero of config space */
    /* bits 0-2 of devid are the function bits */
    err = BestGenericPCIDWGet(slotnumber << 3, 0UL, &value);

    if (B_E_OK == err)
    {
      if ((value == LookForValue) && (cNumFound++ == number))
      {
        /* we found the right BEST ... **SUCCESS** */
        *devid = slotnumber << 3;
        return B_E_OK;
      }
    }
    else if (B_E_NO_BEST_PCI_DEVICE_FOUND == err)
    {
      /* do nothing...this is an appropriate response */
    }
    else
    {
      /* a real error...return it */
      return err;
    }
  }

  /* we didn't find the one the caller was looking for */
  *devid = 0xffffffffUL;
  return B_E_NO_BEST_PCI_DEVICE_FOUND;

#endif /* defined WINNT */
}

/******************************************************************************
 * PCI Config Mailbox Register Functions
 *****************************************************************************/

/*---------------------------------------------------------------------------*
 * b_errtype EXPORT BestPCICfgMailboxReceiveRegRead(
 *
 * Purpose	: read value from mailbox register via PCI
 * Inputs	: devid
 * Outputs	: value, status
 * Returns	: B_E_OK on success, B_E_<errcode> otherwise
 *---------------------------------------------------------------------------*/
b_errtype EXPORT BestPCICfgMailboxReceiveRegRead(
    b_int32 devid,
    b_int32 * value,
    b_int32 * status
)
{
  b_errtype err;
  err = BestPCIMailboxRead(devid, value);

  *status = (err == B_E_OK) ? 0x1 : 0;

  if (err)
    *value = 0xffffffffUL;

  /* B_ERRETURN not possible */
  return B_E_OK;
}


/*---------------------------------------------------------------------------*
 * b_errtype EXPORT BestPCICfgMailboxSendRegWrite(
 *
 * Purpose	: write value to mailbox register via PCI
 * Inputs	: devid, value
 * Outputs	: status
 * Returns	: B_E_OK on success, B_E_<errcode> otherwise
 *---------------------------------------------------------------------------*/
b_errtype EXPORT BestPCICfgMailboxSendRegWrite(
    b_int32 devid,
    b_int32 value,
    b_int32 * status
)
{
  b_errtype err = BestPCIMailboxWrite(devid, value);
  *status = (err == B_E_OK) ? 0x01 : 0;
  /* B_ERRETURN not possible */
  return B_E_OK;
}

#endif /* def BEST_FIRMWARE */
